/*!
 * Font Awesome Icon Picker
 * https://itsjavi.com/fontawesome-iconpicker/
 *
 * Originally written by (c) 2016 Javi Aguilar
 * Licensed under the MIT License
 * https://github.com/itsjavi/fontawesome-iconpicker/blob/master/LICENSE
 *
 */

(function(factory) {
        "use strict";
        if (typeof define === 'function' && define.amd) {
            define(['jquery'], factory);
        } else if (window.jQuery && !window.jQuery.fn.iconpicker) {
            factory(window.jQuery);
        }
    }
    (function($) {
        'use strict';

        var _helpers = {
            isEmpty: function(val) {
                return ((val === false) || (val === '') || (val === null) || (val === undefined));
            },
            isEmptyObject: function(val) {
                return (this.isEmpty(val) === true) || (val.length === 0);
            },
            isElement: function(selector) {
                return ($(selector).length > 0);
            },
            isString: function(val) {
                return ((typeof val === 'string') || (val instanceof String));
            },
            isArray: function(val) {
                return $.isArray(val);
            },
            inArray: function(val, arr) {
                return ($.inArray(val, arr) !== -1);
            },
            throwError: function(text) {
                throw "Font Awesome Icon Picker Exception: " + text;
            }
        };

        var Iconpicker = function(element, options) {
            this._id = Iconpicker._idCounter++;
            this.element = $(element).addClass('iconpicker-element');
            this._trigger('iconpickerCreate');
            this.options = $.extend({}, Iconpicker.defaultOptions, this.element.data(), options);
            this.options.templates = $.extend({}, Iconpicker.defaultOptions.templates, this.options.templates);
            this.options.originalPlacement = this.options.placement;

            // Iconpicker container element
            this.container = (_helpers.isElement(this.options.container) ? $(this.options.container) : false);
            if (this.container === false) {
                if (this.element.is('.dropdown-toggle')) {
                    this.container = $('~ .dropdown-menu:first', this.element);
                } else {
                    this.container = (this.element.is('input,textarea,button,.btn') ? this.element.parent() : this.element);
                }
            }
            this.container.addClass('iconpicker-container');

            if (this.isDropdownMenu()) {
                // if you try to click the dropdown, it is closed, because of that
                // we'll hide some picker controls
                this.options.templates.search = false;
                this.options.templates.buttons = false;
                this.options.placement = 'inline';
            }

            // Is the element an input? Should we search inside for any input?     
            this.input = (this.element.is('input,textarea') ? this.element.addClass('iconpicker-input') : false);
            if (this.input === false) {
                this.input = (this.container.find(this.options.input));
                if (!this.input.is('input,textarea')) {
                    this.input = false;
                }
            }

            // Plugin as component ?
            this.component = this.isDropdownMenu() ? this.container.parent().find(this.options.component) : this.container.find(this.options.component);
            if (this.component.length === 0) {
                this.component = false;
            } else {
                this.component.find('i').addClass('iconpicker-component');
            }

            // Create popover and iconpicker HTML
            this._createPopover();
            this._createIconpicker();

            if (this.getAcceptButton().length === 0) {
                // disable this because we don't have accept buttons
                this.options.mustAccept = false;
            }

            // Avoid CSS issues with input-group-addon(s)
            if (this.isInputGroup()) {
                this.container.parent().append(this.popover);
            } else {
                this.container.append(this.popover);
            }

            // Bind events
            this._bindElementEvents();
            this._bindWindowEvents();

            // Refresh everything
            this.update(this.options.selected);

            if (this.isInline()) {
                this.show();
            }

            this._trigger('iconpickerCreated');
        };

        // Instance identifier counter
        Iconpicker._idCounter = 0;

        Iconpicker.defaultOptions = {
            title: false, // Popover title (optional) only if specified in the template
            selected: false, // use this value as the current item and ignore the original
            defaultValue: false, // use this value as the current item if input or element value is empty
            placement: 'bottom', // (has some issues with auto and CSS). auto, top, bottom, left, right
            collision: 'none', // If true, the popover will be repositioned to another position when collapses with the window borders
            animation: true, // fade in/out on show/hide ?
            //hide iconpicker automatically when a value is picked. it is ignored if mustAccept is not false and the accept button is visible
            hideOnSelect: false,
            showFooter: false,
            searchInFooter: false, // If true, the search will be added to the footer instead of the title
            mustAccept: false, // only applicable when there's an iconpicker-btn-accept button in the popover footer
            selectedCustomClass: 'bg-primary', // Appends this class when to the selected item
            icons: [], // list of icon classes (declared at the bottom of this script for maintainability)
            fullClassFormatter: function(val) {
                return 'fa ' + val;
            },
            input: 'input,.iconpicker-input', // children input selector
            inputSearch: false, // use the input as a search box too?
            container: false, //  Appends the popover to a specific element. If not set, the selected element or element parent is used
            component: '.input-group-addon,.iconpicker-component', // children component jQuery selector or object, relative to the container element
            // Plugin templates:
            templates: {
                popover: '<div class="iconpicker-popover popover"><div class="arrow"></div>' +
                    '<div class="popover-title"></div><div class="popover-content"></div></div>',
                footer: '<div class="popover-footer"></div>',
                buttons: '<button class="iconpicker-btn iconpicker-btn-cancel btn btn-default btn-sm">Cancel</button>' +
                    ' <button class="iconpicker-btn iconpicker-btn-accept btn btn-primary btn-sm">Accept</button>',
                search: '<input type="search" class="form-control iconpicker-search" placeholder="Type to filter" />',
                iconpicker: '<div class="iconpicker"><div class="iconpicker-items"></div></div>',
                iconpickerItem: '<a role="button" href="#" class="iconpicker-item"><i></i></a>',
            }
        };

        Iconpicker.batch = function(selector, method) {
            var args = Array.prototype.slice.call(arguments, 2);
            return $(selector).each(function() {
                var $inst = $(this).data('iconpicker');
                if (!!$inst) {
                    $inst[method].apply($inst, args);
                }
            });
        };

        Iconpicker.prototype = {
            constructor: Iconpicker,
            options: {},
            _id: 0, // instance identifier for bind/unbind events
            _trigger: function(name, opts) {
                //triggers an event bound to the element
                opts = opts || {};
                this.element.trigger($.extend({
                    type: name,
                    iconpickerInstance: this
                }, opts));
                //console.log(name + ' triggered for instance #' + this._id);
            },
            _createPopover: function() {
                this.popover = $(this.options.templates.popover);

                // title (header)
                var _title = this.popover.find('.popover-title');
                if (!!this.options.title) {
                    _title.append($('<div class="popover-title-text">' + this.options.title + '</div>'));
                }
                if (this.hasSeparatedSearchInput() && !this.options.searchInFooter) {
                    _title.append(this.options.templates.search);
                } else if (!this.options.title) {
                    _title.remove();
                }

                // footer
                if (this.options.showFooter && !_helpers.isEmpty(this.options.templates.footer)) {
                    var _footer = $(this.options.templates.footer);
                    if (this.hasSeparatedSearchInput() && this.options.searchInFooter) {
                        _footer.append($(this.options.templates.search));
                    }
                    if (!_helpers.isEmpty(this.options.templates.buttons)) {
                        _footer.append($(this.options.templates.buttons));
                    }
                    this.popover.append(_footer);
                }

                if (this.options.animation === true) {
                    this.popover.addClass('fade');
                }

                return this.popover;
            },
            _createIconpicker: function() {
                var _self = this;
                this.iconpicker = $(this.options.templates.iconpicker);

                var itemClickFn = function(e) {
                    var $this = $(this);
                    if ($this.is('i')) {
                        $this = $this.parent();
                    }

                    _self._trigger('iconpickerSelect', {
                        iconpickerItem: $this,
                        iconpickerValue: _self.iconpickerValue
                    });

                    if (_self.options.mustAccept === false) {
                        _self.update($this.data('iconpickerValue'));
                        _self._trigger('iconpickerSelected', {
                            iconpickerItem: this,
                            iconpickerValue: _self.iconpickerValue
                        });
                    } else {
                        _self.update($this.data('iconpickerValue'), true);
                    }

                    if (_self.options.hideOnSelect && (_self.options.mustAccept === false)) {
                        // only hide when the accept button is not present
                        _self.hide();
                    }
                    e.preventDefault();
                    return false;
                };

                for (var i in this.options.icons) {
                    if (typeof this.options.icons[i] === 'string') {

	                    var itemElement = $(this.options.templates.iconpickerItem);

	                    if( this.options.icons[i].indexOf('kingdom-') == 0 ){
                    		itemElement.find('i')
	                            .addClass( 'kingdom-micon ' + this.options.icons[i] );
                    	}else{
	                        itemElement.find('i')
	                            .addClass(this.options.fullClassFormatter(this.options.icons[i]));
	                    }


	                        itemElement.data('iconpickerValue', this.options.icons[i])
	                            .on('click.iconpicker', itemClickFn);
	                        this.iconpicker.find('.iconpicker-items').append(itemElement
	                            .attr('title', '.' + this.options.icons[i]));
                    }
                }

                this.popover.find('.popover-content').append(this.iconpicker);

                return this.iconpicker;
            },
            _isEventInsideIconpicker: function(e) {
                var _t = $(e.target);
                if ((!_t.hasClass('iconpicker-element')  ||
                        (_t.hasClass('iconpicker-element') && !_t.is(this.element))) &&
                    (_t.parents('.iconpicker-popover').length === 0)) {
                    return false;
                }
                return true;
            },
            _bindElementEvents: function() {
                var _self = this;

                this.getSearchInput().on('keyup.iconpicker', function() {
                    _self.filter($(this).val().toLowerCase());
                });

                this.getAcceptButton().on('click.iconpicker', function() {
                    var _picked = _self.iconpicker.find('.iconpicker-selected').get(0);

                    _self.update(_self.iconpickerValue);

                    _self._trigger('iconpickerSelected', {
                        iconpickerItem: _picked,
                        iconpickerValue: _self.iconpickerValue
                    });
                    if (!_self.isInline()) {
                        _self.hide();
                    }
                });
                this.getCancelButton().on('click.iconpicker', function() {
                    if (!_self.isInline()) {
                        _self.hide();
                    }
                });

                this.element.on('focus.iconpicker', function(e) {
                    _self.show();
                    e.stopPropagation();
                });

                if (this.hasComponent()) {
                    this.component.on('click.iconpicker', function() {
                        _self.toggle();
                    });
                }

                if (this.hasInput()) {
                    // Bind input keyup event
                    this.input.on('keyup.iconpicker', function(e) {
                        if (!_helpers.inArray(e.keyCode, [38, 40, 37, 39, 16, 17, 18, 9, 8, 91, 93, 20, 46, 186, 190, 46, 78, 188, 44, 86])) {
                            _self.update();
                        } else {
                            _self._updateFormGroupStatus(_self.getValid(this.value) !== false);
                        }
                        if (_self.options.inputSearch === true) {
                            _self.filter($(this).val().toLowerCase());
                        }
                        //_self.hide();
                    });
                }

            },
            _bindWindowEvents: function() {
                var $doc = $(window.document);
                var _self = this;

                // Add a namespace to the document events so they can be identified
                // later for every instance separately
                var _eventNs = '.iconpicker.inst' + this._id;

                $(window).on('resize.iconpicker' + _eventNs + ' orientationchange.iconpicker' + _eventNs, function(e) {
                    // reposition popover
                    if (_self.popover.hasClass('in')) {
                        _self.updatePlacement();
                    }
                });

                if (!_self.isInline()) {
                    $doc.on('mouseup' + _eventNs, function(e) {
                        if (!_self._isEventInsideIconpicker(e) && !_self.isInline()) {
                            _self.hide();
                        }
                        e.stopPropagation();
                        e.preventDefault();
                        return false;
                    });
                }

                return false;
            },
            _unbindElementEvents: function() {
                this.popover.off('.iconpicker');
                this.element.off('.iconpicker');

                if (this.hasInput()) {
                    this.input.off('.iconpicker');
                }

                if (this.hasComponent()) {
                    this.component.off('.iconpicker');
                }

                if (this.hasContainer()) {
                    this.container.off('.iconpicker');
                }
            },
            _unbindWindowEvents: function() {
                // destroy window and window.document bound events
                $(window).off('.iconpicker.inst' + this._id);
                $(window.document).off('.iconpicker.inst' + this._id);
            },
            updatePlacement: function(placement, collision) {
                placement = placement || this.options.placement;
                this.options.placement = placement; // set new placement
                collision = collision || this.options.collision;
                collision = (collision === true ? 'flip' : collision);

                var _pos = {
                    // at: Defines which position (or side) on container element to align the
                    // popover element against: "horizontal vertical" alignment.
                    at: "right bottom",
                    // my: Defines which position (or side) on the popover being positioned to align
                    // with the container element: "horizontal vertical" alignment
                    my: "right top",
                    // of: Which element to position against.
                    of: (this.hasInput() && !this.isInputGroup()) ? this.input : this.container,
                    // collision: When the positioned element overflows the window (or within element) 
                    // in some direction, move it to an alternative position.
                    collision: (collision === true ? 'flip' : collision),
                    // within: Element to position within, affecting collision detection.
                    within: window
                };

                // remove previous classes
                this.popover.removeClass('inline topLeftCorner topLeft top topRight topRightCorner ' +
                    'rightTop right rightBottom bottomRight bottomRightCorner ' +
                    'bottom bottomLeft bottomLeftCorner leftBottom left leftTop');

                if (typeof placement === 'object') {
                    // custom position ?
                    return this.popover.pos($.extend({}, _pos, placement));
                }

                switch (placement) {
                    case 'inline':
                        {
                            _pos = false;
                        }
                        break;
                    case 'topLeftCorner':
                        {
                            _pos.my = 'right bottom';
                            _pos.at = 'left top';
                        }
                        break;

                    case 'topLeft':
                        {
                            _pos.my = 'left bottom';
                            _pos.at = 'left top';
                        }
                        break;

                    case 'top':
                        {
                            _pos.my = 'center bottom';
                            _pos.at = 'center top';
                        }
                        break;

                    case 'topRight':
                        {
                            _pos.my = 'right bottom';
                            _pos.at = 'right top';
                        }
                        break;

                    case 'topRightCorner':
                        {
                            _pos.my = 'left bottom';
                            _pos.at = 'right top';
                        }
                        break;

                    case 'rightTop':
                        {
                            _pos.my = 'left bottom';
                            _pos.at = 'right center';
                        }
                        break;

                    case 'right':
                        {
                            _pos.my = 'left center';
                            _pos.at = 'right center';
                        }
                        break;

                    case 'rightBottom':
                        {
                            _pos.my = 'left top';
                            _pos.at = 'right center';
                        }
                        break;

                    case 'bottomRightCorner':
                        {
                            _pos.my = 'left top';
                            _pos.at = 'right bottom';
                        }
                        break;

                    case 'bottomRight':
                        {
                            _pos.my = 'right top';
                            _pos.at = 'right bottom';
                        }
                        break;
                    case 'bottom':
                        {
                            _pos.my = 'center top';
                            _pos.at = 'center bottom';
                        }
                        break;

                    case 'bottomLeft':
                        {
                            _pos.my = 'left top';
                            _pos.at = 'left bottom';
                        }
                        break;

                    case 'bottomLeftCorner':
                        {
                            _pos.my = 'right top';
                            _pos.at = 'left bottom';
                        }
                        break;

                    case 'leftBottom':
                        {
                            _pos.my = 'right top';
                            _pos.at = 'left center';
                        }
                        break;

                    case 'left':
                        {
                            _pos.my = 'right center';
                            _pos.at = 'left center';
                        }
                        break;

                    case 'leftTop':
                        {
                            _pos.my = 'right bottom';
                            _pos.at = 'left center';
                        }
                        break;

                    default:
                        {
                            return false;
                        }
                        break;

                }

                this.popover.css({
                    'display': (this.options.placement === 'inline') ? '' : 'block'
                });

                if (_pos !== false) {
                   
                    //reset position
                    this.popover.css({
                        'top': '180px',
                        'right': 'auto',
                        'bottom': 'auto',
                        'left': 'auto',
                        'maxWidth': 'none'
                    });
                }
                this.popover.addClass(this.options.placement);

                return true;
            },
            _updateComponents: function() {
                // Update selected item
                this.iconpicker.find('.iconpicker-item.iconpicker-selected')
                    .removeClass('iconpicker-selected ' + this.options.selectedCustomClass);

                if (this.iconpickerValue) {
                    this.iconpicker.find('.' + this.options.fullClassFormatter(this.iconpickerValue).replace(/ /g, '.')).parent()
                        .addClass('iconpicker-selected ' + this.options.selectedCustomClass);
                }

                // Update component item
                if (this.hasComponent()) {
                    var icn = this.component.find('i');
                    if (icn.length > 0) {
                    	
                    	if( this.iconpickerValue.indexOf('kingdom-') == 0 ){
                    		icn.attr('class', 'kingdom-micon ' + this.iconpickerValue );
                    	}else{
	                        icn.attr('class', this.options.fullClassFormatter(this.iconpickerValue));
	                    }
                    } else {
                        this.component.html(this.getHtml());
                    }
                }

            },
            _updateFormGroupStatus: function(isValid) {
                if (this.hasInput()) {
                    if (isValid !== false) {
                        // Remove form-group error class if any
                        this.input.parents('.form-group:first').removeClass('has-error');
                    } else {
                        this.input.parents('.form-group:first').addClass('has-error');
                    }
                    return true;
                }
                return false;
            },
            getValid: function(val) {
                // here we must validate the value (you may change this validation
                // to suit your needs
                if (!_helpers.isString(val)) {
                    val = '';
                }

                var isEmpty = (val === '');

                // trim string
                val = $.trim(val);

                if (_helpers.inArray(val, this.options.icons) || isEmpty) {
                    return val;
                }
                return false;
            },
            /**
             * Sets the internal item value and updates everything, excepting the input or element.
             * For doing so, call setSourceValue() or update() instead
             */
            setValue: function(val) {
                // sanitize first
                var _val = this.getValid(val);
                if (_val !== false) {
                    this.iconpickerValue = _val;
                    this._trigger('iconpickerSetValue', {
                        iconpickerValue: _val
                    });
                    return this.iconpickerValue;
                } else {
                    this._trigger('iconpickerInvalid', {
                        iconpickerValue: val
                    });
                    return false;
                }
            },
            getHtml: function() {
                return '<i class="' + this.options.fullClassFormatter(this.iconpickerValue) + '"></i>';
            },
            /**
             * Calls setValue and if it's a valid item value, sets the input or element value
             */
            setSourceValue: function(val) {
                val = this.setValue(val);
                if ((val !== false) && (val !== '')) {
                    if (this.hasInput()) {
                        this.input.val(this.iconpickerValue);
                    } else {
                        this.element.data('iconpickerValue', this.iconpickerValue);
                    }
                    this._trigger('iconpickerSetSourceValue', {
                        iconpickerValue: val
                    });
                }
                return val;
            },
            /**
             * Returns the input or element item value, without formatting, or defaultValue
             * if it's empty string, undefined, false or null
             * @param {type} defaultValue
             * @returns string|mixed
             */
            getSourceValue: function(defaultValue) {
                // returns the input or element value, as string
                defaultValue = defaultValue || this.options.defaultValue;
                var val = defaultValue;

                if (this.hasInput()) {
                    val = this.input.val();
                } else {
                    val = this.element.data('iconpickerValue');
                }
                if ((val === undefined) || (val === '') || (val === null) || (val === false)) {
                    // if not defined or empty, return default
                    val = defaultValue;
                }
                return val;
            },
            hasInput: function() {
                return (this.input !== false);
            },
            isInputSearch: function() {
                return (this.hasInput() && (this.options.inputSearch === true));
            },
            isInputGroup: function() {
                return this.container.is('.input-group');
            },
            isDropdownMenu: function() {
                return this.container.is('.dropdown-menu');
            },
            hasSeparatedSearchInput: function() {
                return (this.options.templates.search !== false) && (!this.isInputSearch());
            },
            hasComponent: function() {
                return (this.component !== false);
            },
            hasContainer: function() {
                return (this.container !== false);
            },
            getAcceptButton: function() {
                return this.popover.find('.iconpicker-btn-accept');
            },
            getCancelButton: function() {
                return this.popover.find('.iconpicker-btn-cancel');
            },
            getSearchInput: function() {
                return this.popover.find('.iconpicker-search');
            },
            filter: function(filterText) {
                if (_helpers.isEmpty(filterText)) {
                    this.iconpicker.find('.iconpicker-item').show();
                    return $(false);
                } else {
                    var found = [];
                    this.iconpicker.find('.iconpicker-item').each(function() {
                        var $this = $(this);
                        var text = $this.attr('title').toLowerCase();
                        var regex = false;
                        try {
                            regex = new RegExp(filterText, 'g');
                        } catch (e) {
                            regex = false;
                        }
                        if ((regex !== false) && text.match(regex)) {
                            found.push($this);
                            $this.show();
                        } else {
                            $this.hide();
                        }
                    });
                    return found;
                }
            },
            show: function() {
                if (this.popover.hasClass('in')) {
                    return false;
                }
                // hide other non-inline pickers
                $.iconpicker.batch($('.iconpicker-popover.in:not(.inline)').not(this.popover), 'hide');

                this._trigger('iconpickerShow');
                this.updatePlacement();
                this.popover.addClass('in');
                setTimeout($.proxy(function() {
                    this.popover.css('display', this.isInline() ? '' : 'block');
                    this._trigger('iconpickerShown');
                }, this), this.options.animation ? 300 : 1); // animation duration
            },
            hide: function() {
                if (!this.popover.hasClass('in')) {
                    return false;
                }
                this._trigger('iconpickerHide');
                this.popover.removeClass('in');
                setTimeout($.proxy(function() {
                    this.popover.css('display', 'none');
                    this.getSearchInput().val('');
                    this.filter(''); // clear filter
                    this._trigger('iconpickerHidden');
                }, this), this.options.animation ? 300 : 1);
            },
            toggle: function() {
                if (this.popover.is(":visible")) {
                    this.hide();
                } else {
                    this.show(true);
                }
            },
            update: function(val, updateOnlyInternal) {
                val = (val ? val :  this.getSourceValue(this.iconpickerValue));
                // reads the input or element value again and tries to update the plugin
                // fallback to the current selected item value
                this._trigger('iconpickerUpdate');

                if (updateOnlyInternal === true) {
                    val = this.setValue(val);
                } else {
                    val = this.setSourceValue(val);
                    this._updateFormGroupStatus(val !== false);
                }

                if (val !== false) {
                    this._updateComponents();
                }

                this._trigger('iconpickerUpdated');
                return val;
            },
            destroy: function() {
                this._trigger('iconpickerDestroy');

                // unbinds events and resets everything to the initial state,
                // including component mode
                this.element.removeData('iconpicker').removeData('iconpickerValue').removeClass('iconpicker-element');

                this._unbindElementEvents();
                this._unbindWindowEvents();

                $(this.popover).remove();

                this._trigger('iconpickerDestroyed');
            },
            disable: function() {
                if (this.hasInput()) {
                    this.input.prop('disabled', true);
                    return true;
                }
                return false;
            },
            enable: function() {
                if (this.hasInput()) {
                    this.input.prop('disabled', false);
                    return true;
                }
                return false;
            },
            isDisabled: function() {
                if (this.hasInput()) {
                    return (this.input.prop('disabled') === true);
                }
                return false;
            },
            isInline: function() {
                return (this.options.placement === 'inline') || (this.popover.hasClass('inline'));
            }
        };

        $.iconpicker = Iconpicker;

        // jQuery plugin
        $.fn.iconpicker = function(options) {
            return this.each(function() {
                var $this = $(this);
                if (!$this.data('iconpicker')) {
                    // create plugin instance (only if not exists) and expose the entire instance API
                    $this.data('iconpicker', new Iconpicker(this, ((typeof options === 'object') ? options : {})));
                }
            });
        };

        // List of all Font Awesome icons without class prefix
        Iconpicker.defaultOptions.icons = [
             /* ultimate moon icon set */
            'kingdom-icon-home', 'kingdom-icon-home2', 'kingdom-icon-home3', 'kingdom-icon-home4', 'kingdom-icon-home5', 'kingdom-icon-home6', 'kingdom-icon-home7', 'kingdom-icon-home8', 'kingdom-icon-home9', 'kingdom-icon-office', 'kingdom-icon-city', 'kingdom-icon-newspaper', 'kingdom-icon-magazine', 'kingdom-icon-design', 'kingdom-icon-pencil', 'kingdom-icon-pencil2', 'kingdom-icon-pencil3', 'kingdom-icon-pencil4', 'kingdom-icon-pencil5', 'kingdom-icon-pencil6', 'kingdom-icon-pencil7', 'kingdom-icon-eraser', 'kingdom-icon-eraser2', 'kingdom-icon-eraser3', 'kingdom-icon-quill', 'kingdom-icon-quill2', 'kingdom-icon-quill3', 'kingdom-icon-quill4', 'kingdom-icon-pen', 'kingdom-icon-pen-plus', 'kingdom-icon-pen-minus', 'kingdom-icon-pen2', 'kingdom-icon-pen3', 'kingdom-icon-pen4', 'kingdom-icon-pen5', 'kingdom-icon-marker', 'kingdom-icon-marker2', 'kingdom-icon-blog', 'kingdom-icon-blog2', 'kingdom-icon-pen6', 'kingdom-icon-brush', 'kingdom-icon-spray', 'kingdom-icon-palette', 'kingdom-icon-palette2', 'kingdom-icon-color-sampler', 'kingdom-icon-toggle', 'kingdom-icon-bucket', 'kingdom-icon-bucket2', 'kingdom-icon-gradient', 'kingdom-icon-eyedropper', 'kingdom-icon-eyedropper2', 'kingdom-icon-eyedropper3', 'kingdom-icon-droplet', 'kingdom-icon-droplet2', 'kingdom-icon-droplet-contrast', 'kingdom-icon-color-clear', 'kingdom-icon-paint-format', 'kingdom-icon-stamp', 'kingdom-icon-images', 'kingdom-icon-image', 'kingdom-icon-image2', 'kingdom-icon-image-compare', 'kingdom-icon-images2', 'kingdom-icon-image3', 'kingdom-icon-images3', 'kingdom-icon-image4', 'kingdom-icon-image5', 'kingdom-icon-image6', 'kingdom-icon-camera', 'kingdom-icon-camera2', 'kingdom-icon-camera3', 'kingdom-icon-shutter', 'kingdom-icon-headphones', 'kingdom-icon-headset', 'kingdom-icon-music', 'kingdom-icon-music2', 'kingdom-icon-music3', 'kingdom-icon-music4', 'kingdom-icon-music5', 'kingdom-icon-music6', 'kingdom-icon-album', 'kingdom-icon-cassette', 'kingdom-icon-tape', 'kingdom-icon-piano', 'kingdom-icon-guitar', 'kingdom-icon-trumpet', 'kingdom-icon-speakers', 'kingdom-icon-play', 'kingdom-icon-play2', 'kingdom-icon-clapboard-play', 'kingdom-icon-clapboard', 'kingdom-icon-media', 'kingdom-icon-presentation', 'kingdom-icon-movie', 'kingdom-icon-film', 'kingdom-icon-film2', 'kingdom-icon-film3', 'kingdom-icon-film4', 'kingdom-icon-video-camera', 'kingdom-icon-video-camera2', 'kingdom-icon-video-camera-slash', 'kingdom-icon-video-camera3', 'kingdom-icon-video-camera4', 'kingdom-icon-video-camera-slash2', 'kingdom-icon-video-camera5', 'kingdom-icon-recording', 'kingdom-icon-dice', 'kingdom-icon-gamepad', 'kingdom-icon-gamepad2', 'kingdom-icon-gamepad3', 'kingdom-icon-pacman', 'kingdom-icon-spades', 'kingdom-icon-clubs', 'kingdom-icon-diamonds', 'kingdom-icon-chess-king', 'kingdom-icon-chess-queen', 'kingdom-icon-chess-rock', 'kingdom-icon-chess-bishop', 'kingdom-icon-chess-knight', 'kingdom-icon-chess-pawn', 'kingdom-icon-chess', 'kingdom-icon-bullhorn', 'kingdom-icon-megaphone', 'kingdom-icon-new', 'kingdom-icon-connection', 'kingdom-icon-connection2', 'kingdom-icon-connection3', 'kingdom-icon-connection4', 'kingdom-icon-podcast', 'kingdom-icon-station', 'kingdom-icon-station2', 'kingdom-icon-antenna', 'kingdom-icon-satellite-dish', 'kingdom-icon-satellite-dish2', 'kingdom-icon-feed', 'kingdom-icon-podcast2', 'kingdom-icon-antenna2', 'kingdom-icon-mic', 'kingdom-icon-mic-off', 'kingdom-icon-mic2', 'kingdom-icon-mic-off2', 'kingdom-icon-mic3', 'kingdom-icon-mic4', 'kingdom-icon-mic5', 'kingdom-icon-book', 'kingdom-icon-book2', 'kingdom-icon-book-play', 'kingdom-icon-book3', 'kingdom-icon-bookmark', 'kingdom-icon-books', 'kingdom-icon-archive', 'kingdom-icon-reading', 'kingdom-icon-library', 'kingdom-icon-library2', 'kingdom-icon-graduation', 'kingdom-icon-file-text', 'kingdom-icon-profile', 'kingdom-icon-file-empty', 'kingdom-icon-file-empty2', 'kingdom-icon-files-empty', 'kingdom-icon-files-empty2', 'kingdom-icon-file-plus', 'kingdom-icon-file-plus2', 'kingdom-icon-file-minus', 'kingdom-icon-file-minus2', 'kingdom-icon-file-download', 'kingdom-icon-file-download2', 'kingdom-icon-file-upload', 'kingdom-icon-file-upload2', 'kingdom-icon-file-check', 'kingdom-icon-file-check2', 'kingdom-icon-file-eye', 'kingdom-icon-file-eye2', 'kingdom-icon-file-text2', 'kingdom-icon-file-text3', 'kingdom-icon-file-picture', 'kingdom-icon-file-picture2', 'kingdom-icon-file-music', 'kingdom-icon-file-music2', 'kingdom-icon-file-play', 'kingdom-icon-file-play2', 'kingdom-icon-file-video', 'kingdom-icon-file-video2', 'kingdom-icon-copy', 'kingdom-icon-copy2', 'kingdom-icon-file-zip', 'kingdom-icon-file-zip2', 'kingdom-icon-file-xml', 'kingdom-icon-file-xml2', 'kingdom-icon-file-css', 'kingdom-icon-file-css2', 'kingdom-icon-file-presentation', 'kingdom-icon-file-presentation2', 'kingdom-icon-file-stats', 'kingdom-icon-file-stats2', 'kingdom-icon-file-locked', 'kingdom-icon-file-locked2', 'kingdom-icon-file-spreadsheet', 'kingdom-icon-file-spreadsheet2', 'kingdom-icon-copy3', 'kingdom-icon-copy4', 'kingdom-icon-paste', 'kingdom-icon-paste2', 'kingdom-icon-paste3', 'kingdom-icon-paste4', 'kingdom-icon-stack', 'kingdom-icon-stack2', 'kingdom-icon-stack3', 'kingdom-icon-folder', 'kingdom-icon-folder-search', 'kingdom-icon-folder-download', 'kingdom-icon-folder-upload', 'kingdom-icon-folder-plus', 'kingdom-icon-folder-plus2', 'kingdom-icon-folder-minus', 'kingdom-icon-folder-minus2', 'kingdom-icon-folder-check', 'kingdom-icon-folder-heart', 'kingdom-icon-folder-remove', 'kingdom-icon-folder2', 'kingdom-icon-folder-open', 'kingdom-icon-folder3', 'kingdom-icon-folder4', 'kingdom-icon-folder-plus3', 'kingdom-icon-folder-minus3', 'kingdom-icon-folder-plus4', 'kingdom-icon-folder-minus4', 'kingdom-icon-folder-download2', 'kingdom-icon-folder-upload2', 'kingdom-icon-folder-download3', 'kingdom-icon-folder-upload3', 'kingdom-icon-folder5', 'kingdom-icon-folder-open2', 'kingdom-icon-folder6', 'kingdom-icon-folder-open3', 'kingdom-icon-certificate', 'kingdom-icon-cc', 'kingdom-icon-price-tag', 'kingdom-icon-price-tag2', 'kingdom-icon-price-tags', 'kingdom-icon-price-tag3', 'kingdom-icon-price-tags2', 'kingdom-icon-price-tag4', 'kingdom-icon-price-tags3', 'kingdom-icon-price-tag5', 'kingdom-icon-price-tags4', 'kingdom-icon-barcode', 'kingdom-icon-barcode2', 'kingdom-icon-qrcode', 'kingdom-icon-ticket', 'kingdom-icon-ticket2', 'kingdom-icon-theater', 'kingdom-icon-store', 'kingdom-icon-store2', 'kingdom-icon-cart', 'kingdom-icon-cart2', 'kingdom-icon-cart3', 'kingdom-icon-cart4', 'kingdom-icon-cart5', 'kingdom-icon-cart-add', 'kingdom-icon-cart-add2', 'kingdom-icon-cart-remove', 'kingdom-icon-basket', 'kingdom-icon-bag', 'kingdom-icon-bag2', 'kingdom-icon-percent', 'kingdom-icon-coins', 'kingdom-icon-coin-dollar', 'kingdom-icon-coin-euro', 'kingdom-icon-coin-pound', 'kingdom-icon-coin-yen', 'kingdom-icon-piggy-bank', 'kingdom-icon-wallet', 'kingdom-icon-cash', 'kingdom-icon-cash2', 'kingdom-icon-cash3', 'kingdom-icon-cash4', 'kingdom-icon-credit-card', 'kingdom-icon-credit-card2', 'kingdom-icon-calculator', 'kingdom-icon-calculator2', 'kingdom-icon-calculator3', 'kingdom-icon-chip', 'kingdom-icon-lifebuoy', 'kingdom-icon-phone', 'kingdom-icon-phone2', 'kingdom-icon-phone-slash', 'kingdom-icon-phone-wave', 'kingdom-icon-phone-plus', 'kingdom-icon-phone-minus', 'kingdom-icon-phone-plus2', 'kingdom-icon-phone-minus2', 'kingdom-icon-phone-incoming', 'kingdom-icon-phone-outgoing', 'kingdom-icon-phone3', 'kingdom-icon-phone-slash2', 'kingdom-icon-phone-wave2', 'kingdom-icon-phone-hang-up', 'kingdom-icon-phone-hang-up2', 'kingdom-icon-address-book', 'kingdom-icon-address-book2', 'kingdom-icon-address-book3', 'kingdom-icon-notebook', 'kingdom-icon-at-sign', 'kingdom-icon-envelop', 'kingdom-icon-envelop2', 'kingdom-icon-envelop3', 'kingdom-icon-envelop4', 'kingdom-icon-envelop5', 'kingdom-icon-mailbox', 'kingdom-icon-pushpin', 'kingdom-icon-pushpin2', 'kingdom-icon-location', 'kingdom-icon-location2', 'kingdom-icon-location3', 'kingdom-icon-location4', 'kingdom-icon-location5', 'kingdom-icon-location6', 'kingdom-icon-location7', 'kingdom-icon-compass', 'kingdom-icon-compass2', 'kingdom-icon-compass3', 'kingdom-icon-compass4', 'kingdom-icon-compass5', 'kingdom-icon-compass6', 'kingdom-icon-map', 'kingdom-icon-map2', 'kingdom-icon-map3', 'kingdom-icon-map4', 'kingdom-icon-map5', 'kingdom-icon-direction', 'kingdom-icon-reset', 'kingdom-icon-history', 'kingdom-icon-clock', 'kingdom-icon-clock2', 'kingdom-icon-clock3', 'kingdom-icon-clock4', 'kingdom-icon-watch', 'kingdom-icon-watch2', 'kingdom-icon-alarm', 'kingdom-icon-alarm-add', 'kingdom-icon-alarm-check', 'kingdom-icon-alarm-cancel', 'kingdom-icon-bell', 'kingdom-icon-bell2', 'kingdom-icon-bell3', 'kingdom-icon-bell-plus', 'kingdom-icon-bell-minus', 'kingdom-icon-bell-check', 'kingdom-icon-bell-cross', 'kingdom-icon-stopwatch', 'kingdom-icon-calendar', 'kingdom-icon-calendar2', 'kingdom-icon-calendar3', 'kingdom-icon-calendar4', 'kingdom-icon-calendar5', 'kingdom-icon-printer', 'kingdom-icon-printer2', 'kingdom-icon-printer3', 'kingdom-icon-printer4', 'kingdom-icon-shredder', 'kingdom-icon-mouse', 'kingdom-icon-mouse2', 'kingdom-icon-mouse-left', 'kingdom-icon-mouse-right', 'kingdom-icon-keyboard', 'kingdom-icon-keyboard2', 'kingdom-icon-typewriter', 'kingdom-icon-display', 'kingdom-icon-display2', 'kingdom-icon-display3', 'kingdom-icon-display4', 'kingdom-icon-laptop', 'kingdom-icon-mobile', 'kingdom-icon-mobile2', 'kingdom-icon-tablet', 'kingdom-icon-mobile3', 'kingdom-icon-tv', 'kingdom-icon-radio', 'kingdom-icon-cabinet', 'kingdom-icon-drawer', 'kingdom-icon-drawer2', 'kingdom-icon-drawer-out', 'kingdom-icon-drawer-in', 'kingdom-icon-drawer3', 'kingdom-icon-box', 'kingdom-icon-box-add', 'kingdom-icon-box-remove', 'kingdom-icon-download', 'kingdom-icon-upload', 'kingdom-icon-floppy-disk', 'kingdom-icon-floppy-disks', 'kingdom-icon-floppy-disks2', 'kingdom-icon-floppy-disk2', 'kingdom-icon-floppy-disk3', 'kingdom-icon-disk', 'kingdom-icon-disk2', 'kingdom-icon-usb-stick', 'kingdom-icon-drive', 'kingdom-icon-server', 'kingdom-icon-database', 'kingdom-icon-database2', 'kingdom-icon-database3', 'kingdom-icon-database4', 'kingdom-icon-database-menu', 'kingdom-icon-database-add', 'kingdom-icon-database-remove', 'kingdom-icon-database-insert', 'kingdom-icon-database-export', 'kingdom-icon-database-upload', 'kingdom-icon-database-refresh', 'kingdom-icon-database-diff', 'kingdom-icon-database-edit', 'kingdom-icon-database-edit2', 'kingdom-icon-database-check', 'kingdom-icon-database-arrow', 'kingdom-icon-database-time', 'kingdom-icon-database-time2', 'kingdom-icon-undo', 'kingdom-icon-redo', 'kingdom-icon-rotate-ccw', 'kingdom-icon-rotate-cw', 'kingdom-icon-rotate-ccw2', 'kingdom-icon-rotate-cw2', 'kingdom-icon-rotate-ccw3', 'kingdom-icon-rotate-cw3', 'kingdom-icon-flip-vertical', 'kingdom-icon-flip-vertical2', 'kingdom-icon-flip-horizontal', 'kingdom-icon-flip-horizontal2', 'kingdom-icon-flip-vertical3', 'kingdom-icon-flip-vertical4', 'kingdom-icon-angle', 'kingdom-icon-shear', 'kingdom-icon-unite', 'kingdom-icon-subtract', 'kingdom-icon-interset', 'kingdom-icon-exclude', 'kingdom-icon-align-left', 'kingdom-icon-align-center-horizontal', 'kingdom-icon-align-right', 'kingdom-icon-align-top', 'kingdom-icon-align-center-vertical', 'kingdom-icon-align-bottom', 'kingdom-icon-undo2', 'kingdom-icon-redo2', 'kingdom-icon-forward', 'kingdom-icon-reply', 'kingdom-icon-reply-all', 'kingdom-icon-bubble', 'kingdom-icon-bubbles', 'kingdom-icon-bubbles2', 'kingdom-icon-bubble2', 'kingdom-icon-bubbles3', 'kingdom-icon-bubbles4', 'kingdom-icon-bubble-notification', 'kingdom-icon-bubbles5', 'kingdom-icon-bubbles6', 'kingdom-icon-bubble3', 'kingdom-icon-bubble-dots', 'kingdom-icon-bubble-lines', 'kingdom-icon-bubble4', 'kingdom-icon-bubble-dots2', 'kingdom-icon-bubble-lines2', 'kingdom-icon-bubble5', 'kingdom-icon-bubble6', 'kingdom-icon-bubbles7', 'kingdom-icon-bubble7', 'kingdom-icon-bubbles8', 'kingdom-icon-bubble8', 'kingdom-icon-bubble-dots3', 'kingdom-icon-bubble-lines3', 'kingdom-icon-bubble9', 'kingdom-icon-bubble-dots4', 'kingdom-icon-bubble-lines4', 'kingdom-icon-bubbles9', 'kingdom-icon-bubbles10', 'kingdom-icon-bubble-blocked', 'kingdom-icon-bubble-quote', 'kingdom-icon-bubble-user', 'kingdom-icon-bubble-smiley', 'kingdom-icon-bubble-check', 'kingdom-icon-bubble-video', 'kingdom-icon-bubble-link', 'kingdom-icon-bubble-lock', 'kingdom-icon-bubble-star', 'kingdom-icon-bubble-heart', 'kingdom-icon-bubble-paperclip', 'kingdom-icon-bubble-cancel', 'kingdom-icon-bubble-plus', 'kingdom-icon-bubble-minus', 'kingdom-icon-bubble-notification2', 'kingdom-icon-bubble-left', 'kingdom-icon-bubble-right', 'kingdom-icon-bubble-first', 'kingdom-icon-bubble-last', 'kingdom-icon-bubble-reply', 'kingdom-icon-bubble-forward', 'kingdom-icon-bubble-reply2', 'kingdom-icon-bubble-forward2', 'kingdom-icon-user', 'kingdom-icon-users', 'kingdom-icon-user-plus', 'kingdom-icon-user-minus', 'kingdom-icon-user-cancel', 'kingdom-icon-user-block', 'kingdom-icon-user-lock', 'kingdom-icon-user-check', 'kingdom-icon-users2', 'kingdom-icon-user2', 'kingdom-icon-users3', 'kingdom-icon-user-plus2', 'kingdom-icon-user-minus2', 'kingdom-icon-user-cancel2', 'kingdom-icon-user-block2', 'kingdom-icon-user-lock2', 'kingdom-icon-user-check2', 'kingdom-icon-user3', 'kingdom-icon-user4', 'kingdom-icon-user5', 'kingdom-icon-user6', 'kingdom-icon-users4', 'kingdom-icon-user-tie', 'kingdom-icon-user7', 'kingdom-icon-users5', 'kingdom-icon-collaboration', 'kingdom-icon-vcard', 'kingdom-icon-hat', 'kingdom-icon-bowtie', 'kingdom-icon-tie', 'kingdom-icon-hanger', 'kingdom-icon-tshirt', 'kingdom-icon-pants', 'kingdom-icon-shorts', 'kingdom-icon-sock', 'kingdom-icon-laundry', 'kingdom-icon-quotes-left', 'kingdom-icon-quotes-right', 'kingdom-icon-quotes-left2', 'kingdom-icon-quotes-right2', 'kingdom-icon-hour-glass', 'kingdom-icon-hour-glass2', 'kingdom-icon-hour-glass3', 'kingdom-icon-spinner', 'kingdom-icon-spinner2', 'kingdom-icon-spinner3', 'kingdom-icon-spinner4', 'kingdom-icon-spinner5', 'kingdom-icon-spinner6', 'kingdom-icon-spinner7', 'kingdom-icon-spinner8', 'kingdom-icon-spinner9', 'kingdom-icon-spinner10', 'kingdom-icon-spinner11', 'kingdom-icon-microscope', 'kingdom-icon-binoculars', 'kingdom-icon-binoculars2', 'kingdom-icon-search', 'kingdom-icon-zoom-in', 'kingdom-icon-zoom-out', 'kingdom-icon-search2', 'kingdom-icon-zoom-in2', 'kingdom-icon-zoom-out2', 'kingdom-icon-search3', 'kingdom-icon-search4', 'kingdom-icon-zoom-in3', 'kingdom-icon-zoom-out3', 'kingdom-icon-search5', 'kingdom-icon-search6', 'kingdom-icon-enlarge', 'kingdom-icon-shrink', 'kingdom-icon-enlarge2', 'kingdom-icon-shrink2', 'kingdom-icon-enlarge3', 'kingdom-icon-shrink3', 'kingdom-icon-enlarge4', 'kingdom-icon-shrink4', 'kingdom-icon-enlarge5', 'kingdom-icon-shrink5', 'kingdom-icon-enlarge6', 'kingdom-icon-shrink6', 'kingdom-icon-enlarge7', 'kingdom-icon-shrink7', 'kingdom-icon-key', 'kingdom-icon-key2', 'kingdom-icon-key3', 'kingdom-icon-key4', 'kingdom-icon-key5', 'kingdom-icon-keyhole', 'kingdom-icon-lock', 'kingdom-icon-lock2', 'kingdom-icon-lock3', 'kingdom-icon-lock4', 'kingdom-icon-unlocked', 'kingdom-icon-lock5', 'kingdom-icon-unlocked2', 'kingdom-icon-safe', 'kingdom-icon-wrench', 'kingdom-icon-wrench2', 'kingdom-icon-wrench3', 'kingdom-icon-equalizer', 'kingdom-icon-equalizer2', 'kingdom-icon-equalizer3', 'kingdom-icon-equalizer4', 'kingdom-icon-cog', 'kingdom-icon-cogs', 'kingdom-icon-cog2', 'kingdom-icon-cog3', 'kingdom-icon-cog4', 'kingdom-icon-cog5', 'kingdom-icon-cog6', 'kingdom-icon-cog7', 'kingdom-icon-factory', 'kingdom-icon-hammer', 'kingdom-icon-hammer-wrench', 'kingdom-icon-knife', 'kingdom-icon-screwdriver', 'kingdom-icon-screwdriver2', 'kingdom-icon-magic-wand', 'kingdom-icon-magic-wand2', 'kingdom-icon-pulse', 'kingdom-icon-pulse2', 'kingdom-icon-aid-kit', 'kingdom-icon-aid-kit2', 'kingdom-icon-stethoscope', 'kingdom-icon-ambulance', 'kingdom-icon-alarm2', 'kingdom-icon-pandage', 'kingdom-icon-pandage2', 'kingdom-icon-bug', 'kingdom-icon-bug2', 'kingdom-icon-syringe', 'kingdom-icon-syringe2', 'kingdom-icon-pill', 'kingdom-icon-construction', 'kingdom-icon-traffic-cone', 'kingdom-icon-traffic-lights', 'kingdom-icon-pie-chart', 'kingdom-icon-pie-chart2', 'kingdom-icon-pie-chart3', 'kingdom-icon-pie-chart4', 'kingdom-icon-pie-chart5', 'kingdom-icon-pie-chart6', 'kingdom-icon-pie-chart7', 'kingdom-icon-stats-dots', 'kingdom-icon-stats-bars', 'kingdom-icon-pie-chart8', 'kingdom-icon-stats-bars2', 'kingdom-icon-stats-bars3', 'kingdom-icon-stats-bars4', 'kingdom-icon-stats-bars5', 'kingdom-icon-stats-bars6', 'kingdom-icon-chart', 'kingdom-icon-stats-growth', 'kingdom-icon-stats-decline', 'kingdom-icon-stats-growth2', 'kingdom-icon-stats-decline2', 'kingdom-icon-stairs-up', 'kingdom-icon-stairs-down', 'kingdom-icon-stairs', 'kingdom-icon-stairs2', 'kingdom-icon-ladder', 'kingdom-icon-rating', 'kingdom-icon-rating2', 'kingdom-icon-rating3', 'kingdom-icon-cake', 'kingdom-icon-podium', 'kingdom-icon-stars', 'kingdom-icon-medal-star', 'kingdom-icon-medal', 'kingdom-icon-medal2', 'kingdom-icon-medal-first', 'kingdom-icon-medal-second', 'kingdom-icon-medal-third', 'kingdom-icon-crown', 'kingdom-icon-trophy', 'kingdom-icon-trophy2', 'kingdom-icon-trophy3', 'kingdom-icon-diamond', 'kingdom-icon-diamond2', 'kingdom-icon-trophy4', 'kingdom-icon-gift', 'kingdom-icon-gift2', 'kingdom-icon-balloon', 'kingdom-icon-pipe', 'kingdom-icon-mustache', 'kingdom-icon-mustache2', 'kingdom-icon-glass', 'kingdom-icon-glass2', 'kingdom-icon-glass3', 'kingdom-icon-glass4', 'kingdom-icon-bottle', 'kingdom-icon-bottle2', 'kingdom-icon-bottle3', 'kingdom-icon-bottle4', 'kingdom-icon-mug', 'kingdom-icon-fork-knife', 'kingdom-icon-spoon-knife', 'kingdom-icon-cook', 'kingdom-icon-plate', 'kingdom-icon-steak', 'kingdom-icon-fish', 'kingdom-icon-chicken', 'kingdom-icon-hamburger', 'kingdom-icon-pizza', 'kingdom-icon-cheese', 'kingdom-icon-bread', 'kingdom-icon-icecream', 'kingdom-icon-milk', 'kingdom-icon-cup', 'kingdom-icon-cup2', 'kingdom-icon-coffee-baen', 'kingdom-icon-coffee', 'kingdom-icon-teapot', 'kingdom-icon-leaf', 'kingdom-icon-leaf2', 'kingdom-icon-apple', 'kingdom-icon-grapes', 'kingdom-icon-cherry', 'kingdom-icon-tree', 'kingdom-icon-tree2', 'kingdom-icon-tree3', 'kingdom-icon-tree4', 'kingdom-icon-cactus', 'kingdom-icon-mountains', 'kingdom-icon-paw', 'kingdom-icon-footprint', 'kingdom-icon-tent', 'kingdom-icon-flower', 'kingdom-icon-flower2', 'kingdom-icon-rocket', 'kingdom-icon-meter', 'kingdom-icon-meter2', 'kingdom-icon-meter-slow', 'kingdom-icon-meter-medium', 'kingdom-icon-meter-fast', 'kingdom-icon-meter3', 'kingdom-icon-hammer2', 'kingdom-icon-balance', 'kingdom-icon-scale', 'kingdom-icon-bomb', 'kingdom-icon-fire', 'kingdom-icon-fire2', 'kingdom-icon-lab', 'kingdom-icon-atom', 'kingdom-icon-atom2', 'kingdom-icon-magnet', 'kingdom-icon-magnet2', 'kingdom-icon-dumbbell', 'kingdom-icon-skull', 'kingdom-icon-skull2', 'kingdom-icon-lamp', 'kingdom-icon-lamp2', 'kingdom-icon-lamp3', 'kingdom-icon-spotlight', 'kingdom-icon-lamp4', 'kingdom-icon-lamp5', 'kingdom-icon-lamp6', 'kingdom-icon-lamp7', 'kingdom-icon-lamp8', 'kingdom-icon-broom', 'kingdom-icon-toiletpaper', 'kingdom-icon-bin', 'kingdom-icon-bin2', 'kingdom-icon-bin3', 'kingdom-icon-bin4', 'kingdom-icon-bin5', 'kingdom-icon-briefcase', 'kingdom-icon-briefcase2', 'kingdom-icon-briefcase3', 'kingdom-icon-airplane', 'kingdom-icon-airplane2', 'kingdom-icon-airplane3', 'kingdom-icon-airplane4', 'kingdom-icon-paperplane', 'kingdom-icon-car', 'kingdom-icon-steering-wheel', 'kingdom-icon-car2', 'kingdom-icon-gas', 'kingdom-icon-bus', 'kingdom-icon-truck', 'kingdom-icon-bike', 'kingdom-icon-road', 'kingdom-icon-train', 'kingdom-icon-train2', 'kingdom-icon-ship', 'kingdom-icon-boat', 'kingdom-icon-chopper', 'kingdom-icon-chopper2', 'kingdom-icon-cube', 'kingdom-icon-cube2', 'kingdom-icon-cube3', 'kingdom-icon-cube4', 'kingdom-icon-pyramid', 'kingdom-icon-pyramid2', 'kingdom-icon-package', 'kingdom-icon-puzzle', 'kingdom-icon-puzzle2', 'kingdom-icon-puzzle3', 'kingdom-icon-puzzle4', 'kingdom-icon-glasses-3d', 'kingdom-icon-glasses-3d2', 'kingdom-icon-glasses', 'kingdom-icon-glasses2', 'kingdom-icon-brain', 'kingdom-icon-accessibility', 'kingdom-icon-accessibility2', 'kingdom-icon-strategy', 'kingdom-icon-target', 'kingdom-icon-target2', 'kingdom-icon-target3', 'kingdom-icon-ammo', 'kingdom-icon-gun', 'kingdom-icon-gun-forbidden', 'kingdom-icon-shield', 'kingdom-icon-shield-check', 'kingdom-icon-shield-notice', 'kingdom-icon-shield2', 'kingdom-icon-shield3', 'kingdom-icon-shield4', 'kingdom-icon-soccer', 'kingdom-icon-football', 'kingdom-icon-archery', 'kingdom-icon-bow', 'kingdom-icon-sword', 'kingdom-icon-tennis', 'kingdom-icon-pingpong', 'kingdom-icon-baseball', 'kingdom-icon-basketball', 'kingdom-icon-golf', 'kingdom-icon-golf2', 'kingdom-icon-hockey', 'kingdom-icon-racing', 'kingdom-icon-finish', 'kingdom-icon-eight-ball', 'kingdom-icon-bowling', 'kingdom-icon-bowling2', 'kingdom-icon-power', 'kingdom-icon-power2', 'kingdom-icon-power3', 'kingdom-icon-switch', 'kingdom-icon-switch2', 'kingdom-icon-power-cord', 'kingdom-icon-power-cord2', 'kingdom-icon-outlet', 'kingdom-icon-clipboard', 'kingdom-icon-clipboard2', 'kingdom-icon-clipboard3', 'kingdom-icon-clipboard4', 'kingdom-icon-clipboard5', 'kingdom-icon-clipboard6', 'kingdom-icon-playlist', 'kingdom-icon-playlist-add', 'kingdom-icon-list-numbered', 'kingdom-icon-list', 'kingdom-icon-list2', 'kingdom-icon-more', 'kingdom-icon-more2', 'kingdom-icon-grid', 'kingdom-icon-grid2', 'kingdom-icon-grid3', 'kingdom-icon-grid4', 'kingdom-icon-grid5', 'kingdom-icon-grid6', 'kingdom-icon-grid7', 'kingdom-icon-tree5', 'kingdom-icon-tree6', 'kingdom-icon-tree7', 'kingdom-icon-lan', 'kingdom-icon-lan2', 'kingdom-icon-lan3', 'kingdom-icon-menu', 'kingdom-icon-circle-small', 'kingdom-icon-menu2', 'kingdom-icon-menu3', 'kingdom-icon-menu4', 'kingdom-icon-menu5', 'kingdom-icon-menu6', 'kingdom-icon-menu7', 'kingdom-icon-menu8', 'kingdom-icon-menu9', 'kingdom-icon-menu10', 'kingdom-icon-cloud', 'kingdom-icon-cloud-download', 'kingdom-icon-cloud-upload', 'kingdom-icon-cloud-check', 'kingdom-icon-cloud2', 'kingdom-icon-cloud-download2', 'kingdom-icon-cloud-upload2', 'kingdom-icon-cloud-check2', 'kingdom-icon-download2', 'kingdom-icon-download3', 'kingdom-icon-upload2', 'kingdom-icon-upload3', 'kingdom-icon-import', 'kingdom-icon-import2', 'kingdom-icon-download4', 'kingdom-icon-upload4', 'kingdom-icon-download5', 'kingdom-icon-upload5', 'kingdom-icon-download6', 'kingdom-icon-upload6', 'kingdom-icon-download7', 'kingdom-icon-upload7', 'kingdom-icon-download8', 'kingdom-icon-upload8', 'kingdom-icon-download9', 'kingdom-icon-upload9', 'kingdom-icon-download10', 'kingdom-icon-upload10', 'kingdom-icon-sphere', 'kingdom-icon-sphere2', 'kingdom-icon-sphere3', 'kingdom-icon-planet', 'kingdom-icon-planet2', 'kingdom-icon-earth', 'kingdom-icon-earth2', 'kingdom-icon-earth3', 'kingdom-icon-link', 'kingdom-icon-unlink', 'kingdom-icon-link2', 'kingdom-icon-unlink2', 'kingdom-icon-link3', 'kingdom-icon-unlink3', 'kingdom-icon-link4', 'kingdom-icon-unlink4', 'kingdom-icon-link5', 'kingdom-icon-unlink5', 'kingdom-icon-anchor', 'kingdom-icon-flag', 'kingdom-icon-flag2', 'kingdom-icon-flag3', 'kingdom-icon-flag4', 'kingdom-icon-flag5', 'kingdom-icon-flag6', 'kingdom-icon-flag7', 'kingdom-icon-flag8', 'kingdom-icon-attachment', 'kingdom-icon-attachment2', 'kingdom-icon-eye', 'kingdom-icon-eye-plus', 'kingdom-icon-eye-minus', 'kingdom-icon-eye-blocked', 'kingdom-icon-eye2', 'kingdom-icon-eye-blocked2', 'kingdom-icon-eye3', 'kingdom-icon-eye-blocked3', 'kingdom-icon-eye4', 'kingdom-icon-bookmark2', 'kingdom-icon-bookmark3', 'kingdom-icon-bookmarks', 'kingdom-icon-bookmark4', 'kingdom-icon-spotlight2', 'kingdom-icon-starburst', 'kingdom-icon-snowflake', 'kingdom-icon-temperature', 'kingdom-icon-temperature2', 'kingdom-icon-weather-lightning', 'kingdom-icon-weather-lightning2', 'kingdom-icon-weather-rain', 'kingdom-icon-weather-rain2', 'kingdom-icon-weather-snow', 'kingdom-icon-weather-snow2', 'kingdom-icon-weather-cloud-wind', 'kingdom-icon-weather-cloud-wind2', 'kingdom-icon-weather-cloud-sun', 'kingdom-icon-weather-cloud-sun2', 'kingdom-icon-weather-cloudy', 'kingdom-icon-weather-cloudy2', 'kingdom-icon-weather-sun-wind', 'kingdom-icon-weather-sun-wind2', 'kingdom-icon-sun', 'kingdom-icon-sun2', 'kingdom-icon-moon', 'kingdom-icon-day-night', 'kingdom-icon-day-night2', 'kingdom-icon-weather-windy', 'kingdom-icon-fan', 'kingdom-icon-umbrella', 'kingdom-icon-sun3', 'kingdom-icon-contrast', 'kingdom-icon-brightness-contrast', 'kingdom-icon-brightness-high', 'kingdom-icon-brightness-medium', 'kingdom-icon-brightness-low', 'kingdom-icon-bed', 'kingdom-icon-bed2', 'kingdom-icon-furniture', 'kingdom-icon-chair', 'kingdom-icon-star-empty', 'kingdom-icon-star-empty2', 'kingdom-icon-star-full', 'kingdom-icon-star-empty3', 'kingdom-icon-star-half', 'kingdom-icon-star-full2', 'kingdom-icon-star', 'kingdom-icon-heart', 'kingdom-icon-heart2', 'kingdom-icon-heart3', 'kingdom-icon-heart4', 'kingdom-icon-heart-broken', 'kingdom-icon-heart5', 'kingdom-icon-heart6', 'kingdom-icon-heart-broken2', 'kingdom-icon-heart7', 'kingdom-icon-heart8', 'kingdom-icon-heart-broken3', 'kingdom-icon-lips', 'kingdom-icon-lips2', 'kingdom-icon-thumbs-up', 'kingdom-icon-thumbs-up2', 'kingdom-icon-thumbs-down', 'kingdom-icon-thumbs-down2', 'kingdom-icon-thumbs-up3', 'kingdom-icon-thumbs-down3', 'kingdom-icon-height', 'kingdom-icon-man', 'kingdom-icon-woman', 'kingdom-icon-man-woman', 'kingdom-icon-male', 'kingdom-icon-female', 'kingdom-icon-peace', 'kingdom-icon-yin-yang', 'kingdom-icon-happy', 'kingdom-icon-happy2', 'kingdom-icon-smile', 'kingdom-icon-smile2', 'kingdom-icon-tongue', 'kingdom-icon-tongue2', 'kingdom-icon-sad', 'kingdom-icon-sad2', 'kingdom-icon-wink', 'kingdom-icon-wink2', 'kingdom-icon-grin', 'kingdom-icon-grin2', 'kingdom-icon-cool', 'kingdom-icon-cool2', 'kingdom-icon-angry', 'kingdom-icon-angry2', 'kingdom-icon-evil', 'kingdom-icon-evil2', 'kingdom-icon-shocked', 'kingdom-icon-shocked2', 'kingdom-icon-baffled', 'kingdom-icon-baffled2', 'kingdom-icon-confused', 'kingdom-icon-confused2', 'kingdom-icon-neutral', 'kingdom-icon-neutral2', 'kingdom-icon-hipster', 'kingdom-icon-hipster2', 'kingdom-icon-wondering', 'kingdom-icon-wondering2', 'kingdom-icon-sleepy', 'kingdom-icon-sleepy2', 'kingdom-icon-frustrated', 'kingdom-icon-frustrated2', 'kingdom-icon-crying', 'kingdom-icon-crying2', 'kingdom-icon-cursor', 'kingdom-icon-cursor2', 'kingdom-icon-lasso', 'kingdom-icon-lasso2', 'kingdom-icon-select', 'kingdom-icon-select2', 'kingdom-icon-point-up', 'kingdom-icon-point-right', 'kingdom-icon-point-down', 'kingdom-icon-point-left', 'kingdom-icon-pointer', 'kingdom-icon-reminder', 'kingdom-icon-drag-left-right', 'kingdom-icon-drag-left', 'kingdom-icon-drag-right', 'kingdom-icon-touch', 'kingdom-icon-multitouch', 'kingdom-icon-touch-zoom', 'kingdom-icon-touch-pinch', 'kingdom-icon-hand', 'kingdom-icon-grab', 'kingdom-icon-stack-empty', 'kingdom-icon-stack-plus', 'kingdom-icon-stack-minus', 'kingdom-icon-stack-star', 'kingdom-icon-stack-picture', 'kingdom-icon-stack-down', 'kingdom-icon-stack-up', 'kingdom-icon-stack-cancel', 'kingdom-icon-stack-check', 'kingdom-icon-stack-text', 'kingdom-icon-stack-clubs', 'kingdom-icon-stack-spades', 'kingdom-icon-stack-hearts', 'kingdom-icon-stack-diamonds', 'kingdom-icon-stack-user', 'kingdom-icon-stack4', 'kingdom-icon-stack-music', 'kingdom-icon-stack-play', 'kingdom-icon-move', 'kingdom-icon-dots', 'kingdom-icon-warning', 'kingdom-icon-warning2', 'kingdom-icon-notification', 'kingdom-icon-notification2', 'kingdom-icon-question', 'kingdom-icon-question2', 'kingdom-icon-question3', 'kingdom-icon-question4', 'kingdom-icon-question5', 'kingdom-icon-question6', 'kingdom-icon-plus', 'kingdom-icon-minus', 'kingdom-icon-plus2', 'kingdom-icon-minus2', 'kingdom-icon-plus3', 'kingdom-icon-minus3', 'kingdom-icon-plus-circle', 'kingdom-icon-minus-circle', 'kingdom-icon-plus-circle2', 'kingdom-icon-minus-circle2', 'kingdom-icon-info', 'kingdom-icon-info2', 'kingdom-icon-cancel-circle', 'kingdom-icon-cancel-circle2', 'kingdom-icon-blocked', 'kingdom-icon-cancel-square', 'kingdom-icon-cancel-square2', 'kingdom-icon-cancel', 'kingdom-icon-spam', 'kingdom-icon-cross', 'kingdom-icon-cross2', 'kingdom-icon-cross3', 'kingdom-icon-checkmark', 'kingdom-icon-checkmark2', 'kingdom-icon-checkmark3', 'kingdom-icon-checkmark4', 'kingdom-icon-checkmark5', 'kingdom-icon-spell-check', 'kingdom-icon-spell-check2', 'kingdom-icon-enter', 'kingdom-icon-exit', 'kingdom-icon-enter2', 'kingdom-icon-exit2', 'kingdom-icon-enter3', 'kingdom-icon-exit3', 'kingdom-icon-wall', 'kingdom-icon-fence', 'kingdom-icon-play3', 'kingdom-icon-pause', 'kingdom-icon-stop', 'kingdom-icon-previous', 'kingdom-icon-next', 'kingdom-icon-backward', 'kingdom-icon-forward2', 'kingdom-icon-play4', 'kingdom-icon-pause2', 'kingdom-icon-stop2', 'kingdom-icon-backward2', 'kingdom-icon-forward3', 'kingdom-icon-first', 'kingdom-icon-last', 'kingdom-icon-previous2', 'kingdom-icon-next2', 'kingdom-icon-eject', 'kingdom-icon-volume-high', 'kingdom-icon-volume-medium', 'kingdom-icon-volume-low', 'kingdom-icon-volume-mute', 'kingdom-icon-speaker-left', 'kingdom-icon-speaker-right', 'kingdom-icon-volume-mute2', 'kingdom-icon-volume-increase', 'kingdom-icon-volume-decrease', 'kingdom-icon-volume-high2', 'kingdom-icon-volume-medium2', 'kingdom-icon-volume-medium3', 'kingdom-icon-volume-low2', 'kingdom-icon-volume-mute3', 'kingdom-icon-volume-increase2', 'kingdom-icon-volume-decrease2', 'kingdom-icon-volume-decrease3', 'kingdom-icon-volume-5', 'kingdom-icon-volume-4', 'kingdom-icon-volume-3', 'kingdom-icon-volume-2', 'kingdom-icon-volume-1', 'kingdom-icon-volume-0', 'kingdom-icon-volume-mute4', 'kingdom-icon-volume-mute5', 'kingdom-icon-loop', 'kingdom-icon-loop2', 'kingdom-icon-loop3', 'kingdom-icon-infinite-square', 'kingdom-icon-infinite', 'kingdom-icon-infinite2', 'kingdom-icon-loop4', 'kingdom-icon-shuffle', 'kingdom-icon-shuffle2', 'kingdom-icon-wave', 'kingdom-icon-wave2', 'kingdom-icon-split', 'kingdom-icon-merge', 'kingdom-icon-first2', 'kingdom-icon-last2', 'kingdom-icon-arrow-up', 'kingdom-icon-arrow-up2', 'kingdom-icon-arrow-up3', 'kingdom-icon-arrow-up4', 'kingdom-icon-arrow-right', 'kingdom-icon-arrow-right2', 'kingdom-icon-arrow-right3', 'kingdom-icon-arrow-right4', 'kingdom-icon-arrow-down', 'kingdom-icon-arrow-down2', 'kingdom-icon-arrow-down3', 'kingdom-icon-arrow-down4', 'kingdom-icon-arrow-left', 'kingdom-icon-arrow-left2', 'kingdom-icon-arrow-left3', 'kingdom-icon-arrow-left4', 'kingdom-icon-arrow-up5', 'kingdom-icon-arrow-right5', 'kingdom-icon-arrow-down5', 'kingdom-icon-arrow-left5', 'kingdom-icon-arrow-up-left', 'kingdom-icon-arrow-up6', 'kingdom-icon-arrow-up-right', 'kingdom-icon-arrow-right6', 'kingdom-icon-arrow-down-right', 'kingdom-icon-arrow-down6', 'kingdom-icon-arrow-down-left', 'kingdom-icon-arrow-left6', 'kingdom-icon-arrow-up-left2', 'kingdom-icon-arrow-up7', 'kingdom-icon-arrow-up-right2', 'kingdom-icon-arrow-right7', 'kingdom-icon-arrow-down-right2', 'kingdom-icon-arrow-down7', 'kingdom-icon-arrow-down-left2', 'kingdom-icon-arrow-left7', 'kingdom-icon-arrow-up-left3', 'kingdom-icon-arrow-up8', 'kingdom-icon-arrow-up-right3', 'kingdom-icon-arrow-right8', 'kingdom-icon-arrow-down-right3', 'kingdom-icon-arrow-down8', 'kingdom-icon-arrow-down-left3', 'kingdom-icon-arrow-left8', 'kingdom-icon-circle-up', 'kingdom-icon-circle-right', 'kingdom-icon-circle-down', 'kingdom-icon-circle-left', 'kingdom-icon-circle-up2', 'kingdom-icon-circle-right2', 'kingdom-icon-circle-down2', 'kingdom-icon-circle-left2', 'kingdom-icon-circle-up-left', 'kingdom-icon-circle-up3', 'kingdom-icon-circle-up-right', 'kingdom-icon-circle-right3', 'kingdom-icon-circle-down-right', 'kingdom-icon-circle-down3', 'kingdom-icon-circle-down-left', 'kingdom-icon-circle-left3', 'kingdom-icon-circle-up-left2', 'kingdom-icon-circle-up4', 'kingdom-icon-circle-up-right2', 'kingdom-icon-circle-right4', 'kingdom-icon-circle-down-right2', 'kingdom-icon-circle-down4', 'kingdom-icon-circle-down-left2', 'kingdom-icon-circle-left4', 'kingdom-icon-arrow-resize', 'kingdom-icon-arrow-resize2', 'kingdom-icon-arrow-resize3', 'kingdom-icon-arrow-resize4', 'kingdom-icon-arrow-resize5', 'kingdom-icon-arrow-resize6', 'kingdom-icon-arrow-resize7', 'kingdom-icon-arrow-resize8', 'kingdom-icon-square-up-left', 'kingdom-icon-square-up', 'kingdom-icon-square-up-right', 'kingdom-icon-square-right', 'kingdom-icon-square-down-right', 'kingdom-icon-square-down', 'kingdom-icon-square-down-left', 'kingdom-icon-square-left', 'kingdom-icon-arrow-up9', 'kingdom-icon-arrow-right9', 'kingdom-icon-arrow-down9', 'kingdom-icon-arrow-left9', 'kingdom-icon-arrow-up10', 'kingdom-icon-arrow-right10', 'kingdom-icon-arrow-down10', 'kingdom-icon-arrow-left10', 'kingdom-icon-arrow-up-left4', 'kingdom-icon-arrow-up11', 'kingdom-icon-arrow-up-right4', 'kingdom-icon-arrow-right11', 'kingdom-icon-arrow-down-right4', 'kingdom-icon-arrow-down11', 'kingdom-icon-arrow-down-left4', 'kingdom-icon-arrow-left11', 'kingdom-icon-arrow-up-left5', 'kingdom-icon-arrow-up12', 'kingdom-icon-arrow-up-right5', 'kingdom-icon-arrow-right12', 'kingdom-icon-arrow-down-right5', 'kingdom-icon-arrow-down12', 'kingdom-icon-arrow-down-left5', 'kingdom-icon-arrow-left12', 'kingdom-icon-arrow-up13', 'kingdom-icon-arrow-right13', 'kingdom-icon-arrow-down13', 'kingdom-icon-arrow-left13', 'kingdom-icon-arrow-up14', 'kingdom-icon-arrow-right14', 'kingdom-icon-arrow-down14', 'kingdom-icon-arrow-left14', 'kingdom-icon-circle-up5', 'kingdom-icon-circle-right5', 'kingdom-icon-circle-down5', 'kingdom-icon-circle-left5', 'kingdom-icon-circle-up6', 'kingdom-icon-circle-right6', 'kingdom-icon-circle-down6', 'kingdom-icon-circle-left6', 'kingdom-icon-arrow-up15', 'kingdom-icon-arrow-right15', 'kingdom-icon-arrow-down15', 'kingdom-icon-arrow-left15', 'kingdom-icon-arrow-up16', 'kingdom-icon-arrow-right16', 'kingdom-icon-arrow-down16', 'kingdom-icon-arrow-left16', 'kingdom-icon-menu-open', 'kingdom-icon-menu-open2', 'kingdom-icon-menu-close', 'kingdom-icon-menu-close2', 'kingdom-icon-enter4', 'kingdom-icon-enter5', 'kingdom-icon-esc', 'kingdom-icon-enter6', 'kingdom-icon-backspace', 'kingdom-icon-backspace2', 'kingdom-icon-tab', 'kingdom-icon-transmission', 'kingdom-icon-transmission2', 'kingdom-icon-sort', 'kingdom-icon-sort2', 'kingdom-icon-move-up', 'kingdom-icon-move-down', 'kingdom-icon-sort-alpha-asc', 'kingdom-icon-sort-alpha-desc', 'kingdom-icon-sort-numeric-asc', 'kingdom-icon-sort-numberic-desc', 'kingdom-icon-sort-amount-asc', 'kingdom-icon-sort-amount-desc', 'kingdom-icon-sort-time-asc', 'kingdom-icon-sort-time-desc', 'kingdom-icon-battery-6', 'kingdom-icon-battery-5', 'kingdom-icon-battery-4', 'kingdom-icon-battery-3', 'kingdom-icon-battery-2', 'kingdom-icon-battery-1', 'kingdom-icon-battery-0', 'kingdom-icon-battery-charging', 'kingdom-icon-key-keyboard', 'kingdom-icon-key-A', 'kingdom-icon-key-up', 'kingdom-icon-key-right', 'kingdom-icon-key-down', 'kingdom-icon-key-left', 'kingdom-icon-command', 'kingdom-icon-shift', 'kingdom-icon-ctrl', 'kingdom-icon-opt', 'kingdom-icon-checkbox-checked', 'kingdom-icon-checkbox-unchecked', 'kingdom-icon-checkbox-partial', 'kingdom-icon-square', 'kingdom-icon-triangle', 'kingdom-icon-triangle2', 'kingdom-icon-diamond3', 'kingdom-icon-diamond4', 'kingdom-icon-checkbox-checked2', 'kingdom-icon-checkbox-unchecked2', 'kingdom-icon-checkbox-partial2', 'kingdom-icon-radio-checked', 'kingdom-icon-radio-checked2', 'kingdom-icon-radio-unchecked', 'kingdom-icon-toggle-on', 'kingdom-icon-toggle-off', 'kingdom-icon-checkmark-circle', 'kingdom-icon-circle', 'kingdom-icon-circle2', 'kingdom-icon-circles', 'kingdom-icon-circles2', 'kingdom-icon-crop', 'kingdom-icon-crop2', 'kingdom-icon-make-group', 'kingdom-icon-ungroup', 'kingdom-icon-vector', 'kingdom-icon-vector2', 'kingdom-icon-rulers', 'kingdom-icon-ruler', 'kingdom-icon-pencil-ruler', 'kingdom-icon-scissors', 'kingdom-icon-scissors2', 'kingdom-icon-scissors3', 'kingdom-icon-haircut', 'kingdom-icon-comb', 'kingdom-icon-filter', 'kingdom-icon-filter2', 'kingdom-icon-filter3', 'kingdom-icon-filter4', 'kingdom-icon-font', 'kingdom-icon-ampersand', 'kingdom-icon-ligature', 'kingdom-icon-ligature2', 'kingdom-icon-font-size', 'kingdom-icon-typography', 'kingdom-icon-text-height', 'kingdom-icon-text-width', 'kingdom-icon-height2', 'kingdom-icon-width', 'kingdom-icon-bold', 'kingdom-icon-underline', 'kingdom-icon-italic', 'kingdom-icon-strikethrough', 'kingdom-icon-strikethrough2', 'kingdom-icon-font-size2', 'kingdom-icon-bold2', 'kingdom-icon-underline2', 'kingdom-icon-italic2', 'kingdom-icon-strikethrough3', 'kingdom-icon-omega', 'kingdom-icon-sigma', 'kingdom-icon-nbsp', 'kingdom-icon-page-break', 'kingdom-icon-page-break2', 'kingdom-icon-superscript', 'kingdom-icon-subscript', 'kingdom-icon-superscript2', 'kingdom-icon-subscript2', 'kingdom-icon-text-color', 'kingdom-icon-highlight', 'kingdom-icon-pagebreak', 'kingdom-icon-clear-formatting', 'kingdom-icon-table', 'kingdom-icon-table2', 'kingdom-icon-insert-template', 'kingdom-icon-pilcrow', 'kingdom-icon-ltr', 'kingdom-icon-rtl', 'kingdom-icon-ltr2', 'kingdom-icon-rtl2', 'kingdom-icon-section', 'kingdom-icon-paragraph-left', 'kingdom-icon-paragraph-center', 'kingdom-icon-paragraph-right', 'kingdom-icon-paragraph-justify', 'kingdom-icon-paragraph-left2', 'kingdom-icon-paragraph-center2', 'kingdom-icon-paragraph-right2', 'kingdom-icon-paragraph-justify2', 'kingdom-icon-indent-increase', 'kingdom-icon-indent-decrease', 'kingdom-icon-paragraph-left3', 'kingdom-icon-paragraph-center3', 'kingdom-icon-paragraph-right3', 'kingdom-icon-paragraph-justify3', 'kingdom-icon-indent-increase2', 'kingdom-icon-indent-decrease2', 'kingdom-icon-share', 'kingdom-icon-share2', 'kingdom-icon-new-tab', 'kingdom-icon-new-tab2', 'kingdom-icon-popout', 'kingdom-icon-embed', 'kingdom-icon-embed2', 'kingdom-icon-markup', 'kingdom-icon-regexp', 'kingdom-icon-regexp2', 'kingdom-icon-code', 'kingdom-icon-circle-css', 'kingdom-icon-circle-code', 'kingdom-icon-terminal', 'kingdom-icon-unicode', 'kingdom-icon-seven-segment-0', 'kingdom-icon-seven-segment-1', 'kingdom-icon-seven-segment-2', 'kingdom-icon-seven-segment-3', 'kingdom-icon-seven-segment-4', 'kingdom-icon-seven-segment-5', 'kingdom-icon-seven-segment-6', 'kingdom-icon-seven-segment-7', 'kingdom-icon-seven-segment-8', 'kingdom-icon-seven-segment-9', 'kingdom-icon-share3', 'kingdom-icon-share4', 'kingdom-icon-mail', 'kingdom-icon-mail2', 'kingdom-icon-mail3', 'kingdom-icon-mail4', 'kingdom-icon-google', 'kingdom-icon-google-plus', 'kingdom-icon-google-plus2', 'kingdom-icon-google-plus3', 'kingdom-icon-google-drive', 'kingdom-icon-facebook', 'kingdom-icon-facebook2', 'kingdom-icon-facebook3', 'kingdom-icon-ello', 'kingdom-icon-instagram', 'kingdom-icon-twitter', 'kingdom-icon-twitter2', 'kingdom-icon-twitter3', 'kingdom-icon-feed2', 'kingdom-icon-feed3', 'kingdom-icon-feed4', 'kingdom-icon-youtube', 'kingdom-icon-youtube2', 'kingdom-icon-youtube3', 'kingdom-icon-youtube4', 'kingdom-icon-twitch', 'kingdom-icon-vimeo', 'kingdom-icon-vimeo2', 'kingdom-icon-vimeo3', 'kingdom-icon-lanyrd', 'kingdom-icon-flickr', 'kingdom-icon-flickr2', 'kingdom-icon-flickr3', 'kingdom-icon-flickr4', 'kingdom-icon-picassa', 'kingdom-icon-picassa2', 'kingdom-icon-dribbble', 'kingdom-icon-dribbble2', 'kingdom-icon-dribbble3', 'kingdom-icon-forrst', 'kingdom-icon-forrst2', 'kingdom-icon-deviantart', 'kingdom-icon-deviantart2', 'kingdom-icon-steam', 'kingdom-icon-steam2', 'kingdom-icon-dropbox', 'kingdom-icon-onedrive', 'kingdom-icon-github', 'kingdom-icon-github2', 'kingdom-icon-github3', 'kingdom-icon-github4', 'kingdom-icon-github5', 'kingdom-icon-wordpress', 'kingdom-icon-wordpress2', 'kingdom-icon-joomla', 'kingdom-icon-blogger', 'kingdom-icon-blogger2', 'kingdom-icon-tumblr', 'kingdom-icon-tumblr2', 'kingdom-icon-yahoo', 'kingdom-icon-tux', 'kingdom-icon-apple2', 'kingdom-icon-finder', 'kingdom-icon-android', 'kingdom-icon-windows', 'kingdom-icon-windows8', 'kingdom-icon-soundcloud', 'kingdom-icon-soundcloud2', 'kingdom-icon-skype', 'kingdom-icon-reddit', 'kingdom-icon-linkedin', 'kingdom-icon-linkedin2', 'kingdom-icon-lastfm', 'kingdom-icon-lastfm2', 'kingdom-icon-delicious', 'kingdom-icon-stumbleupon', 'kingdom-icon-stumbleupon2', 'kingdom-icon-stackoverflow', 'kingdom-icon-pinterest', 'kingdom-icon-pinterest2', 'kingdom-icon-xing', 'kingdom-icon-xing2', 'kingdom-icon-flattr', 'kingdom-icon-foursquare', 'kingdom-icon-paypal', 'kingdom-icon-paypal2', 'kingdom-icon-paypal3', 'kingdom-icon-yelp', 'kingdom-icon-file-pdf', 'kingdom-icon-file-openoffice', 'kingdom-icon-file-word', 'kingdom-icon-file-excel', 'kingdom-icon-libreoffice', 'kingdom-icon-html5', 'kingdom-icon-html52', 'kingdom-icon-css3', 'kingdom-icon-git', 'kingdom-icon-svg', 'kingdom-icon-codepen', 'kingdom-icon-chrome', 'kingdom-icon-firefox', 'kingdom-icon-IE', 'kingdom-icon-opera', 'kingdom-icon-safari', 'kingdom-icon-IcoMoon',

            'fa-500px', 'fa-address-book', 'fa-address-book-o', 'fa-address-card', 'fa-address-card-o', 'fa-adjust', 'fa-adn',
            'fa-align-center', 'fa-align-justify', 'fa-align-left', 'fa-align-right', 'fa-amazon', 'fa-ambulance',
            'fa-american-sign-language-interpreting', 'fa-anchor', 'fa-android', 'fa-angellist', 'fa-angle-double-down',
            'fa-angle-double-left', 'fa-angle-double-right', 'fa-angle-double-up', 'fa-angle-down', 'fa-angle-left', 'fa-angle-right',
            'fa-angle-up', 'fa-apple', 'fa-archive', 'fa-area-chart', 'fa-arrow-circle-down', 'fa-arrow-circle-left',
            'fa-arrow-circle-o-down', 'fa-arrow-circle-o-left', 'fa-arrow-circle-o-right', 'fa-arrow-circle-o-up',
            'fa-arrow-circle-right', 'fa-arrow-circle-up', 'fa-arrow-down', 'fa-arrow-left', 'fa-arrow-right', 'fa-arrow-up', 'fa-arrows',
            'fa-arrows-alt', 'fa-arrows-h', 'fa-arrows-v', 'fa-asl-interpreting', 'fa-assistive-listening-systems', 'fa-asterisk',
            'fa-at', 'fa-audio-description', 'fa-automobile', 'fa-backward', 'fa-balance-scale', 'fa-ban', 'fa-bandcamp', 'fa-bank',
            'fa-bar-chart', 'fa-bar-chart-o', 'fa-barcode', 'fa-bars', 'fa-bath', 'fa-bathtub', 'fa-battery', 'fa-battery-0',
            'fa-battery-1', 'fa-battery-2', 'fa-battery-3', 'fa-battery-4', 'fa-battery-empty', 'fa-battery-full', 'fa-battery-half',
            'fa-battery-quarter', 'fa-battery-three-quarters', 'fa-bed', 'fa-beer', 'fa-behance', 'fa-behance-square', 'fa-bell',
            'fa-bell-o', 'fa-bell-slash', 'fa-bell-slash-o', 'fa-bicycle', 'fa-binoculars', 'fa-birthday-cake', 'fa-bitbucket',
            'fa-bitbucket-square', 'fa-bitcoin', 'fa-black-tie', 'fa-blind', 'fa-bluetooth', 'fa-bluetooth-b', 'fa-bold', 'fa-bolt',
            'fa-bomb', 'fa-book', 'fa-bookmark', 'fa-bookmark-o', 'fa-braille', 'fa-briefcase', 'fa-btc', 'fa-bug', 'fa-building',
            'fa-building-o', 'fa-bullhorn', 'fa-bullseye', 'fa-bus', 'fa-buysellads', 'fa-cab', 'fa-calculator', 'fa-calendar',
            'fa-calendar-check-o', 'fa-calendar-minus-o', 'fa-calendar-o', 'fa-calendar-plus-o', 'fa-calendar-times-o', 'fa-camera',
            'fa-camera-retro', 'fa-car', 'fa-caret-down', 'fa-caret-left', 'fa-caret-right', 'fa-caret-square-o-down',
            'fa-caret-square-o-left', 'fa-caret-square-o-right', 'fa-caret-square-o-up', 'fa-caret-up', 'fa-cart-arrow-down',
            'fa-cart-plus', 'fa-cc', 'fa-cc-amex', 'fa-cc-diners-club', 'fa-cc-discover', 'fa-cc-jcb', 'fa-cc-mastercard', 'fa-cc-paypal',
            'fa-cc-stripe', 'fa-cc-visa', 'fa-certificate', 'fa-chain', 'fa-chain-broken', 'fa-check', 'fa-check-circle',
            'fa-check-circle-o', 'fa-check-square', 'fa-check-square-o', 'fa-chevron-circle-down', 'fa-chevron-circle-left',
            'fa-chevron-circle-right', 'fa-chevron-circle-up', 'fa-chevron-down', 'fa-chevron-left', 'fa-chevron-right', 'fa-chevron-up',
            'fa-child', 'fa-chrome', 'fa-circle', 'fa-circle-o', 'fa-circle-o-notch', 'fa-circle-thin', 'fa-clipboard', 'fa-clock-o',
            'fa-clone', 'fa-close', 'fa-cloud', 'fa-cloud-download', 'fa-cloud-upload', 'fa-cny', 'fa-code', 'fa-code-fork', 'fa-codepen',
            'fa-codiepie', 'fa-coffee', 'fa-cog', 'fa-cogs', 'fa-columns', 'fa-comment', 'fa-comment-o', 'fa-commenting',
            'fa-commenting-o', 'fa-comments', 'fa-comments-o', 'fa-compass', 'fa-compress', 'fa-connectdevelop', 'fa-contao', 'fa-copy',
            'fa-copyright', 'fa-creative-commons', 'fa-credit-card', 'fa-credit-card-alt', 'fa-crop', 'fa-crosshairs', 'fa-css3',
            'fa-cube', 'fa-cubes', 'fa-cut', 'fa-cutlery', 'fa-dashboard', 'fa-dashcube', 'fa-database', 'fa-deaf', 'fa-deafness',
            'fa-dedent', 'fa-delicious', 'fa-desktop', 'fa-deviantart', 'fa-diamond', 'fa-digg', 'fa-dollar', 'fa-dot-circle-o',
            'fa-download', 'fa-dribbble', 'fa-drivers-license', 'fa-drivers-license-o', 'fa-dropbox', 'fa-drupal', 'fa-edge', 'fa-edit',
            'fa-eercast', 'fa-eject', 'fa-ellipsis-h', 'fa-ellipsis-v', 'fa-empire', 'fa-envelope', 'fa-envelope-o', 'fa-envelope-open',
            'fa-envelope-open-o', 'fa-envelope-square', 'fa-envira', 'fa-eraser', 'fa-etsy', 'fa-eur', 'fa-euro', 'fa-exchange',
            'fa-exclamation', 'fa-exclamation-circle', 'fa-exclamation-triangle', 'fa-expand', 'fa-expeditedssl', 'fa-external-link',
            'fa-external-link-square', 'fa-eye', 'fa-eye-slash', 'fa-eyedropper', 'fa-fa', 'fa-facebook', 'fa-facebook-f',
            'fa-facebook-official', 'fa-facebook-square', 'fa-fast-backward', 'fa-fast-forward', 'fa-fax', 'fa-feed', 'fa-female',
            'fa-fighter-jet', 'fa-file', 'fa-file-archive-o', 'fa-file-audio-o', 'fa-file-code-o', 'fa-file-excel-o', 'fa-file-image-o',
            'fa-file-movie-o', 'fa-file-o', 'fa-file-pdf-o', 'fa-file-photo-o', 'fa-file-picture-o', 'fa-file-powerpoint-o',
            'fa-file-sound-o', 'fa-file-text', 'fa-file-text-o', 'fa-file-video-o', 'fa-file-word-o', 'fa-file-zip-o', 'fa-files-o',
            'fa-film', 'fa-filter', 'fa-fire', 'fa-fire-extinguisher', 'fa-firefox', 'fa-first-order', 'fa-flag', 'fa-flag-checkered',
            'fa-flag-o', 'fa-flash', 'fa-flask', 'fa-flickr', 'fa-floppy-o', 'fa-folder', 'fa-folder-o', 'fa-folder-open',
            'fa-folder-open-o', 'fa-font', 'fa-font-awesome', 'fa-fonticons', 'fa-fort-awesome', 'fa-forumbee', 'fa-forward',
            'fa-foursquare', 'fa-free-code-camp', 'fa-frown-o', 'fa-futbol-o', 'fa-gamepad', 'fa-gavel', 'fa-gbp', 'fa-ge', 'fa-gear',
            'fa-gears', 'fa-genderless', 'fa-get-pocket', 'fa-gg', 'fa-gg-circle', 'fa-gift', 'fa-git', 'fa-git-square', 'fa-github',
            'fa-github-alt', 'fa-github-square', 'fa-gitlab', 'fa-gittip', 'fa-glass', 'fa-glide', 'fa-glide-g', 'fa-globe', 'fa-google',
            'fa-google-plus', 'fa-google-plus-circle', 'fa-google-plus-official', 'fa-google-plus-square', 'fa-google-wallet',
            'fa-graduation-cap', 'fa-gratipay', 'fa-grav', 'fa-group', 'fa-h-square', 'fa-hacker-news', 'fa-hand-grab-o',
            'fa-hand-lizard-o', 'fa-hand-o-down', 'fa-hand-o-left', 'fa-hand-o-right', 'fa-hand-o-up', 'fa-hand-paper-o',
            'fa-hand-peace-o', 'fa-hand-pointer-o', 'fa-hand-rock-o', 'fa-hand-scissors-o', 'fa-hand-spock-o', 'fa-hand-stop-o',
            'fa-handshake-o', 'fa-hard-of-hearing', 'fa-hashtag', 'fa-hdd-o', 'fa-header', 'fa-headphones', 'fa-heart', 'fa-heart-o',
            'fa-heartbeat', 'fa-history', 'fa-home', 'fa-hospital-o', 'fa-hotel', 'fa-hourglass', 'fa-hourglass-1', 'fa-hourglass-2',
            'fa-hourglass-3', 'fa-hourglass-end', 'fa-hourglass-half', 'fa-hourglass-o', 'fa-hourglass-start', 'fa-houzz', 'fa-html5',
            'fa-i-cursor', 'fa-id-badge', 'fa-id-card', 'fa-id-card-o', 'fa-ils', 'fa-image', 'fa-imdb', 'fa-inbox', 'fa-indent',
            'fa-industry', 'fa-info', 'fa-info-circle', 'fa-inr', 'fa-instagram', 'fa-institution', 'fa-internet-explorer', 'fa-intersex',
            'fa-ioxhost', 'fa-italic', 'fa-joomla', 'fa-jpy', 'fa-jsfiddle', 'fa-key', 'fa-keyboard-o', 'fa-krw', 'fa-language',
            'fa-laptop', 'fa-lastfm', 'fa-lastfm-square', 'fa-leaf', 'fa-leanpub', 'fa-legal', 'fa-lemon-o', 'fa-level-down',
            'fa-level-up', 'fa-life-bouy', 'fa-life-buoy', 'fa-life-ring', 'fa-life-saver', 'fa-lightbulb-o', 'fa-line-chart', 'fa-link',
            'fa-linkedin', 'fa-linkedin-square', 'fa-linode', 'fa-linux', 'fa-list', 'fa-list-alt', 'fa-list-ol', 'fa-list-ul',
            'fa-location-arrow', 'fa-lock', 'fa-long-arrow-down', 'fa-long-arrow-left', 'fa-long-arrow-right', 'fa-long-arrow-up',
            'fa-low-vision', 'fa-magic', 'fa-magnet', 'fa-mail-forward', 'fa-mail-reply', 'fa-mail-reply-all', 'fa-male', 'fa-map',
            'fa-map-marker', 'fa-map-o', 'fa-map-pin', 'fa-map-signs', 'fa-mars', 'fa-mars-double', 'fa-mars-stroke', 'fa-mars-stroke-h',
            'fa-mars-stroke-v', 'fa-maxcdn', 'fa-meanpath', 'fa-medium', 'fa-medkit', 'fa-meetup', 'fa-meh-o', 'fa-mercury',
            'fa-microchip', 'fa-microphone', 'fa-microphone-slash', 'fa-minus', 'fa-minus-circle', 'fa-minus-square', 'fa-minus-square-o',
            'fa-mixcloud', 'fa-mobile', 'fa-mobile-phone', 'fa-modx', 'fa-money', 'fa-moon-o', 'fa-mortar-board', 'fa-motorcycle',
            'fa-mouse-pointer', 'fa-music', 'fa-navicon', 'fa-neuter', 'fa-newspaper-o', 'fa-object-group', 'fa-object-ungroup',
            'fa-odnoklassniki', 'fa-odnoklassniki-square', 'fa-opencart', 'fa-openid', 'fa-opera', 'fa-optin-monster', 'fa-outdent',
            'fa-pagelines', 'fa-paint-brush', 'fa-paper-plane', 'fa-paper-plane-o', 'fa-paperclip', 'fa-paragraph', 'fa-paste',
            'fa-pause', 'fa-pause-circle', 'fa-pause-circle-o', 'fa-paw', 'fa-paypal', 'fa-pencil', 'fa-pencil-square',
            'fa-pencil-square-o', 'fa-percent', 'fa-phone', 'fa-phone-square', 'fa-photo', 'fa-picture-o', 'fa-pie-chart',
            'fa-pied-piper', 'fa-pied-piper-alt', 'fa-pied-piper-pp', 'fa-pinterest', 'fa-pinterest-p', 'fa-pinterest-square', 'fa-plane',
            'fa-play', 'fa-play-circle', 'fa-play-circle-o', 'fa-plug', 'fa-plus', 'fa-plus-circle', 'fa-plus-square', 'fa-plus-square-o',
            'fa-podcast', 'fa-power-off', 'fa-print', 'fa-product-hunt', 'fa-puzzle-piece', 'fa-qq', 'fa-qrcode', 'fa-question',
            'fa-question-circle', 'fa-question-circle-o', 'fa-quora', 'fa-quote-left', 'fa-quote-right', 'fa-ra', 'fa-random',
            'fa-ravelry', 'fa-rebel', 'fa-recycle', 'fa-reddit', 'fa-reddit-alien', 'fa-reddit-square', 'fa-refresh', 'fa-registered',
            'fa-remove', 'fa-renren', 'fa-reorder', 'fa-repeat', 'fa-reply', 'fa-reply-all', 'fa-resistance', 'fa-retweet', 'fa-rmb',
            'fa-road', 'fa-rocket', 'fa-rotate-left', 'fa-rotate-right', 'fa-rouble', 'fa-rss', 'fa-rss-square', 'fa-rub', 'fa-ruble',
            'fa-rupee', 'fa-s15', 'fa-safari', 'fa-save', 'fa-scissors', 'fa-scribd', 'fa-search', 'fa-search-minus', 'fa-search-plus',
            'fa-sellsy', 'fa-send', 'fa-send-o', 'fa-server', 'fa-share', 'fa-share-alt', 'fa-share-alt-square', 'fa-share-square',
            'fa-share-square-o', 'fa-shekel', 'fa-sheqel', 'fa-shield', 'fa-ship', 'fa-shirtsinbulk', 'fa-shopping-bag',
            'fa-shopping-basket', 'fa-shopping-cart', 'fa-shower', 'fa-sign-in', 'fa-sign-language', 'fa-sign-out', 'fa-signal',
            'fa-signing', 'fa-simplybuilt', 'fa-sitemap', 'fa-skyatlas', 'fa-skype', 'fa-slack', 'fa-sliders', 'fa-slideshare',
            'fa-smile-o', 'fa-snapchat', 'fa-snapchat-ghost', 'fa-snapchat-square', 'fa-snowflake-o', 'fa-soccer-ball-o', 'fa-sort',
            'fa-sort-alpha-asc', 'fa-sort-alpha-desc', 'fa-sort-amount-asc', 'fa-sort-amount-desc', 'fa-sort-asc', 'fa-sort-desc',
            'fa-sort-down', 'fa-sort-numeric-asc', 'fa-sort-numeric-desc', 'fa-sort-up', 'fa-soundcloud', 'fa-space-shuttle',
            'fa-spinner', 'fa-spoon', 'fa-spotify', 'fa-square', 'fa-square-o', 'fa-stack-exchange', 'fa-stack-overflow', 'fa-star',
            'fa-star-half', 'fa-star-half-empty', 'fa-star-half-full', 'fa-star-half-o', 'fa-star-o', 'fa-steam', 'fa-steam-square',
            'fa-step-backward', 'fa-step-forward', 'fa-stethoscope', 'fa-sticky-note', 'fa-sticky-note-o', 'fa-stop', 'fa-stop-circle',
            'fa-stop-circle-o', 'fa-street-view', 'fa-strikethrough', 'fa-stumbleupon', 'fa-stumbleupon-circle', 'fa-subscript',
            'fa-subway', 'fa-suitcase', 'fa-sun-o', 'fa-superpowers', 'fa-superscript', 'fa-support', 'fa-table', 'fa-tablet',
            'fa-tachometer', 'fa-tag', 'fa-tags', 'fa-tasks', 'fa-taxi', 'fa-telegram', 'fa-television', 'fa-tencent-weibo',
            'fa-terminal', 'fa-text-height', 'fa-text-width', 'fa-th', 'fa-th-large', 'fa-th-list', 'fa-themeisle', 'fa-thermometer',
            'fa-thermometer-0', 'fa-thermometer-1', 'fa-thermometer-2', 'fa-thermometer-3', 'fa-thermometer-4', 'fa-thermometer-empty',
            'fa-thermometer-full', 'fa-thermometer-half', 'fa-thermometer-quarter', 'fa-thermometer-three-quarters', 'fa-thumb-tack',
            'fa-thumbs-down', 'fa-thumbs-o-down', 'fa-thumbs-o-up', 'fa-thumbs-up', 'fa-ticket', 'fa-times', 'fa-times-circle',
            'fa-times-circle-o', 'fa-times-rectangle', 'fa-times-rectangle-o', 'fa-tint', 'fa-toggle-down', 'fa-toggle-left',
            'fa-toggle-off', 'fa-toggle-on', 'fa-toggle-right', 'fa-toggle-up', 'fa-trademark', 'fa-train', 'fa-transgender',
            'fa-transgender-alt', 'fa-trash', 'fa-trash-o', 'fa-tree', 'fa-trello', 'fa-tripadvisor', 'fa-trophy', 'fa-truck', 'fa-try',
            'fa-tty', 'fa-tumblr', 'fa-tumblr-square', 'fa-turkish-lira', 'fa-tv', 'fa-twitch', 'fa-twitter', 'fa-twitter-square',
            'fa-umbrella', 'fa-underline', 'fa-undo', 'fa-universal-access', 'fa-university', 'fa-unlink', 'fa-unlock', 'fa-unlock-alt',
            'fa-unsorted', 'fa-upload', 'fa-usb', 'fa-usd', 'fa-user', 'fa-user-circle', 'fa-user-circle-o', 'fa-user-md', 'fa-user-o',
            'fa-user-plus', 'fa-user-secret', 'fa-user-times', 'fa-users', 'fa-vcard', 'fa-vcard-o', 'fa-venus', 'fa-venus-double',
            'fa-venus-mars', 'fa-viacoin', 'fa-viadeo', 'fa-viadeo-square', 'fa-video-camera', 'fa-vimeo', 'fa-vimeo-square', 'fa-vine',
            'fa-vk', 'fa-volume-control-phone', 'fa-volume-down', 'fa-volume-off', 'fa-volume-up', 'fa-warning', 'fa-wechat', 'fa-weibo',
            'fa-weixin', 'fa-whatsapp', 'fa-wheelchair', 'fa-wheelchair-alt', 'fa-wifi', 'fa-wikipedia-w', 'fa-window-close',
            'fa-window-close-o', 'fa-window-maximize', 'fa-window-minimize', 'fa-window-restore', 'fa-windows', 'fa-won', 'fa-wordpress',
            'fa-wpbeginner', 'fa-wpexplorer', 'fa-wpforms', 'fa-wrench', 'fa-xing', 'fa-xing-square', 'fa-y-combinator',
            'fa-y-combinator-square', 'fa-yahoo', 'fa-yc', 'fa-yc-square', 'fa-yelp', 'fa-yen', 'fa-yoast', 'fa-youtube',
            'fa-youtube-play', 'fa-youtube-square',
        ];
    }));